<?php
require_once 'config.php';
requireAdmin();

$conn = getDBConnection();

// Get date range
$period = $_GET['period'] ?? '10';
$start_date = $_GET['start_date'] ?? '';
$end_date = $_GET['end_date'] ?? '';

if ($period === 'custom' && $start_date && $end_date) {
    $date_condition = "AND DATE(created_at) BETWEEN ? AND ?";
    $date_params = [$start_date, $end_date];
} elseif ($period === '10') {
    $date_condition = "AND DATE(created_at) >= DATE_SUB(CURDATE(), INTERVAL 10 DAY)";
    $date_params = [];
} elseif ($period === '15') {
    $date_condition = "AND DATE(created_at) >= DATE_SUB(CURDATE(), INTERVAL 15 DAY)";
    $date_params = [];
} else {
    $date_condition = "";
    $date_params = [];
}

// Get task completion report
$task_query = "SELECT 
    DATE(completed_at) as date,
    COUNT(*) as completed_count
    FROM tasks 
    WHERE status = 'completed' {$date_condition}
    GROUP BY DATE(completed_at)
    ORDER BY date DESC";

$task_stmt = $conn->prepare($task_query);
if (!empty($date_params)) {
    $task_stmt->bind_param("ss", $date_params[0], $date_params[1]);
}
$task_stmt->execute();
$task_report = $task_stmt->get_result()->fetch_all(MYSQLI_ASSOC);

// Get user performance report
$user_query = "SELECT 
    u.id,
    u.full_name,
    u.points,
    COUNT(DISTINCT t.id) as total_tasks,
    COUNT(DISTINCT CASE WHEN t.status = 'completed' THEN t.id END) as completed_tasks,
    COUNT(DISTINCT CASE WHEN t.status = 'completed' AND t.completed_at {$date_condition} THEN t.id END) as period_completed
    FROM users u
    LEFT JOIN tasks t ON t.assigned_to = u.id
    WHERE u.role = 'user'
    GROUP BY u.id
    ORDER BY u.points DESC";

$user_report = $conn->query($user_query)->fetch_all(MYSQLI_ASSOC);

// Get points distribution
$points_query = "SELECT 
    DATE(created_at) as date,
    SUM(CASE WHEN points > 0 THEN points ELSE 0 END) as points_earned,
    SUM(CASE WHEN points < 0 THEN ABS(points) ELSE 0 END) as points_lost
    FROM points_history
    WHERE 1=1 {$date_condition}
    GROUP BY DATE(created_at)
    ORDER BY date DESC";

$points_stmt = $conn->prepare($points_query);
if (!empty($date_params)) {
    $points_stmt->bind_param("ss", $date_params[0], $date_params[1]);
}
$points_stmt->execute();
$points_report = $points_stmt->get_result()->fetch_all(MYSQLI_ASSOC);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, viewport-fit=cover">
    <title>Reports - Task Management System</title>
    <link rel="stylesheet" href="assets/css/style.css">
    <script src="assets/js/theme.js"></script>
    <script src="assets/js/notifications.js"></script>
    <script src="assets/js/mobile-menu.js"></script>
</head>
<body class="mobile-app">
    <?php include 'mobile_header.php'; ?>

    <div class="container">
        <!-- Date Range Filter -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">Select Report Period</h2>
            </div>
            <form method="GET" action="">
                <div class="form-row">
                    <div class="form-group">
                        <label for="period">Period</label>
                        <select id="period" name="period" onchange="toggleCustomDates()">
                            <option value="10" <?php echo $period === '10' ? 'selected' : ''; ?>>Last 10 Days</option>
                            <option value="15" <?php echo $period === '15' ? 'selected' : ''; ?>>Last 15 Days</option>
                            <option value="custom" <?php echo $period === 'custom' ? 'selected' : ''; ?>>Custom Range</option>
                            <option value="all" <?php echo $period === 'all' ? 'selected' : ''; ?>>All Time</option>
                        </select>
                    </div>
                    <div class="form-group" id="custom_dates" style="display: <?php echo $period === 'custom' ? 'block' : 'none'; ?>;">
                        <label for="start_date">Start Date</label>
                        <input type="date" id="start_date" name="start_date" value="<?php echo htmlspecialchars($start_date); ?>">
                    </div>
                    <div class="form-group" id="custom_dates_end" style="display: <?php echo $period === 'custom' ? 'block' : 'none'; ?>;">
                        <label for="end_date">End Date</label>
                        <input type="date" id="end_date" name="end_date" value="<?php echo htmlspecialchars($end_date); ?>">
                    </div>
                </div>
                <button type="submit" class="btn btn-primary">Generate Report</button>
            </form>
        </div>
        
        <!-- Task Completion Report -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">Task Completion Report</h2>
            </div>
            <div class="table-container">
                <table>
                    <thead>
                        <tr>
                            <th>Date</th>
                            <th>Tasks Completed</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($task_report)): ?>
                            <tr>
                                <td colspan="2" class="empty-state">No data for selected period</td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($task_report as $row): ?>
                                <tr>
                                    <td><?php echo date('M d, Y', strtotime($row['date'])); ?></td>
                                    <td><?php echo $row['completed_count']; ?></td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
        
        <!-- User Performance Report -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">User Performance Report</h2>
            </div>
            <div class="table-container">
                <table>
                    <thead>
                        <tr>
                            <th>User</th>
                            <th>Total Points</th>
                            <th>Total Tasks</th>
                            <th>Completed</th>
                            <th>Completion %</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($user_report as $user): 
                            $completion_pct = $user['total_tasks'] > 0 ? round(($user['completed_tasks'] / $user['total_tasks']) * 100, 2) : 0;
                        ?>
                            <tr>
                                <td><?php echo htmlspecialchars($user['full_name']); ?></td>
                                <td><?php echo number_format($user['points'], 2); ?></td>
                                <td><?php echo $user['total_tasks']; ?></td>
                                <td><?php echo $user['completed_tasks']; ?></td>
                                <td>
                                    <div class="progress-bar">
                                        <div class="progress-fill" style="width: <?php echo $completion_pct; ?>%">
                                            <?php echo $completion_pct; ?>%
                                        </div>
                                    </div>
                                </td>
                                <td>
                                    <a href="user_details.php?id=<?php echo $user['id']; ?>" class="btn btn-primary btn-sm">View Details</a>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
        
        <!-- Points Distribution Report -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">Points Distribution Report</h2>
            </div>
            <div class="table-container">
                <table>
                    <thead>
                        <tr>
                            <th>Date</th>
                            <th>Points Earned</th>
                            <th>Points Lost</th>
                            <th>Net Points</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($points_report)): ?>
                            <tr>
                                <td colspan="4" class="empty-state">No data for selected period</td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($points_report as $row): 
                                $net = $row['points_earned'] - $row['points_lost'];
                            ?>
                                <tr>
                                    <td><?php echo date('M d, Y', strtotime($row['date'])); ?></td>
                                    <td style="color: green; font-weight: bold;">+<?php echo number_format($row['points_earned'], 2); ?></td>
                                    <td style="color: red; font-weight: bold;">-<?php echo number_format($row['points_lost'], 2); ?></td>
                                    <td style="font-weight: bold; color: <?php echo $net >= 0 ? 'green' : 'red'; ?>;">
                                        <?php echo $net >= 0 ? '+' : ''; ?><?php echo number_format($net, 2); ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
    
    <script>
        function toggleCustomDates() {
            const period = document.getElementById('period').value;
            const customDates = document.getElementById('custom_dates');
            const customDatesEnd = document.getElementById('custom_dates_end');
            if (period === 'custom') {
                customDates.style.display = 'block';
                customDatesEnd.style.display = 'block';
            } else {
                customDates.style.display = 'none';
                customDatesEnd.style.display = 'none';
            }
        }
    </script>
</body>
</html>
