// Notification Dropdown Functionality
(function() {
    'use strict';

    let notificationDropdown = null;
    let notificationIcon = null;
    let notificationOverlay = null;

    function initNotifications() {
        notificationIcon = document.getElementById('notification-icon');
        notificationDropdown = document.getElementById('notification-dropdown');
        notificationOverlay = document.getElementById('notification-overlay');

        if (!notificationIcon || !notificationDropdown) {
            return;
        }

        // Toggle dropdown on icon click
        notificationIcon.addEventListener('click', function(e) {
            e.stopPropagation();
            e.preventDefault();
            toggleNotificationDropdown();
        });

        // Close dropdown when clicking outside
        if (notificationOverlay) {
            notificationOverlay.addEventListener('click', closeNotificationDropdown);
        }

        // Close button click
        const closeBtn = notificationDropdown.querySelector('.notification-close');
        if (closeBtn) {
            closeBtn.addEventListener('click', function(e) {
                e.stopPropagation();
                closeNotificationDropdown();
            });
        }

        // Click outside to close
        document.addEventListener('click', function(e) {
            if (notificationDropdown && notificationDropdown.classList.contains('show')) {
                if (!notificationDropdown.contains(e.target) && !notificationIcon.contains(e.target)) {
                    closeNotificationDropdown();
                }
            }
        });

        // Use event delegation for notification items (works with dynamic content)
        if (notificationDropdown) {
            notificationDropdown.addEventListener('click', function(e) {
                const item = e.target.closest('.notification-item');
                if (item) {
                    const notificationId = item.dataset.notificationId;
                    if (notificationId && !item.classList.contains('read')) {
                        markAsRead(notificationId, item);
                    }
                }
            });
        }
    }

    function toggleNotificationDropdown() {
        if (!notificationDropdown) return;

        const isOpen = notificationDropdown.classList.contains('show');
        
        if (isOpen) {
            closeNotificationDropdown();
        } else {
            openNotificationDropdown();
        }
    }

    function openNotificationDropdown() {
        if (!notificationDropdown || !notificationOverlay) return;

        notificationDropdown.classList.add('show');
        notificationOverlay.classList.add('show');
        
        // Prevent body scroll when dropdown is open
        document.body.style.overflow = 'hidden';
    }

    function closeNotificationDropdown() {
        if (!notificationDropdown || !notificationOverlay) return;

        notificationDropdown.classList.remove('show');
        notificationOverlay.classList.remove('show');
        
        // Restore body scroll
        document.body.style.overflow = '';
    }

    function markAsRead(notificationId, element) {
        if (!notificationId) return;

        fetch('mark_notification_read.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: 'notification_id=' + encodeURIComponent(notificationId)
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                element.classList.remove('unread');
                element.classList.add('read');
                // Refresh count from server to get accurate count
                refreshNotificationCount();
            }
        })
        .catch(error => {
            console.error('Error marking notification as read:', error);
        });
    }

    function updateNotificationCount() {
        // Update the notification badge count
        const badge = document.querySelector('.notification-badge');
        if (badge) {
            const unreadCount = document.querySelectorAll('.notification-item.unread').length;
            if (unreadCount > 0) {
                badge.textContent = unreadCount > 99 ? '99+' : unreadCount;
                badge.classList.remove('hidden');
            } else {
                badge.classList.add('hidden');
            }
        }
    }

    // Refresh notification count from server
    function refreshNotificationCount() {
        fetch('get_notification_count.php')
            .then(response => response.json())
            .then(data => {
                if (data.success !== undefined) {
                    const badge = document.querySelector('.notification-badge');
                    if (badge) {
                        if (data.count > 0) {
                            badge.textContent = data.count > 99 ? '99+' : data.count;
                            badge.classList.remove('hidden');
                        } else {
                            badge.classList.add('hidden');
                        }
                    }
                }
            })
            .catch(error => {
                console.error('Error fetching notification count:', error);
            });
    }

    // Initialize on DOM ready
    function initialize() {
        // Wait a bit to ensure DOM is fully ready
        setTimeout(function() {
            initNotifications();
        }, 100);
    }

    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initialize);
    } else {
        initialize();
    }

    // Expose functions globally if needed
    window.toggleNotificationDropdown = toggleNotificationDropdown;
    window.closeNotificationDropdown = closeNotificationDropdown;
})();
