<?php
require_once 'config.php';
requireAdmin();

$error = '';
$success = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $task_name = trim($_POST['task_name'] ?? '');
    $task_description = trim($_POST['task_description'] ?? '');
    $task_duration = intval($_POST['task_duration'] ?? 0);
    $task_type = $_POST['task_type'] ?? 'one_time';
    $recurrence_interval = $task_type === 'recurring' ? intval($_POST['recurrence_interval'] ?? 0) : null;
    $priority = $_POST['priority'] ?? 'medium';
    $assigned_to = !empty($_POST['assigned_to']) ? intval($_POST['assigned_to']) : null;
    $due_date = !empty($_POST['due_date']) ? $_POST['due_date'] : null;
    
    if (empty($task_name) || $task_duration <= 0) {
        $error = 'Please fill in all required fields';
    } else {
        $conn = getDBConnection();
        
        // Calculate due date if not provided
        if (!$due_date && $task_duration > 0) {
            $due_date = date('Y-m-d H:i:s', strtotime("+{$task_duration} minutes"));
        }
        
        $next_recurrence = null;
        if ($task_type === 'recurring' && $recurrence_interval) {
            $next_recurrence = date('Y-m-d H:i:s', strtotime("+{$recurrence_interval} days"));
        }
        
        $stmt = $conn->prepare("INSERT INTO tasks (task_name, task_description, task_duration, task_type, recurrence_interval, priority, assigned_to, created_by, due_date, next_recurrence_date) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        $stmt->bind_param("ssisisisss", $task_name, $task_description, $task_duration, $task_type, $recurrence_interval, $priority, $assigned_to, $_SESSION['user_id'], $due_date, $next_recurrence);
        
        if ($stmt->execute()) {
            $task_id = $conn->insert_id;
            
            // Create notification for assigned user
            if ($assigned_to) {
                $notif_stmt = $conn->prepare("INSERT INTO notifications (user_id, task_id, notification_type, message) VALUES (?, ?, 'task_assigned', ?)");
                $message = "New task assigned: {$task_name}";
                $notif_stmt->bind_param("iis", $assigned_to, $task_id, $message);
                $notif_stmt->execute();
            }
            
            logAudit('task_created', 'task', $task_id, "Task created: {$task_name}");
            $success = 'Task created successfully!';
            
            // Clear form
            $_POST = [];
        } else {
            $error = 'Error creating task: ' . $conn->error;
        }
    }
}

// Get all users for assignment
$conn = getDBConnection();
$users = $conn->query("SELECT id, username, full_name FROM users WHERE role = 'user' ORDER BY full_name")->fetch_all(MYSQLI_ASSOC);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, viewport-fit=cover">
    <title>Create Task - Task Management System</title>
    <link rel="stylesheet" href="assets/css/style.css">
    <script src="assets/js/theme.js"></script>
    <script src="assets/js/notifications.js"></script>
    <script src="assets/js/mobile-menu.js"></script>
    <script>
        function toggleRecurrence() {
            const taskType = document.getElementById('task_type').value;
            const recurrenceDiv = document.getElementById('recurrence_div');
            recurrenceDiv.style.display = taskType === 'recurring' ? 'block' : 'none';
        }
    </script>
</head>
<body class="mobile-app">
    <?php include 'mobile_header.php'; ?>

    <div class="container">
        <div class="card">
            <?php if ($error): ?>
                <div class="alert alert-error"><?php echo htmlspecialchars($error); ?></div>
            <?php endif; ?>
            <?php if ($success): ?>
                <div class="alert alert-success"><?php echo htmlspecialchars($success); ?></div>
            <?php endif; ?>
            
            <form method="POST" action="" enctype="multipart/form-data">
                <div class="form-group">
                    <label for="task_name">Task Name *</label>
                    <input type="text" id="task_name" name="task_name" required value="<?php echo htmlspecialchars($_POST['task_name'] ?? ''); ?>">
                </div>
                
                <div class="form-group">
                    <label for="task_description">Task Description</label>
                    <textarea id="task_description" name="task_description" rows="4"><?php echo htmlspecialchars($_POST['task_description'] ?? ''); ?></textarea>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="task_duration">Duration (minutes) *</label>
                        <input type="number" id="task_duration" name="task_duration" min="1" required value="<?php echo htmlspecialchars($_POST['task_duration'] ?? ''); ?>">
                    </div>
                    
                    <div class="form-group">
                        <label for="due_date">Due Date</label>
                        <input type="datetime-local" id="due_date" name="due_date" value="<?php echo htmlspecialchars($_POST['due_date'] ?? ''); ?>">
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="task_type">Task Type *</label>
                        <select id="task_type" name="task_type" required onchange="toggleRecurrence()">
                            <option value="one_time" <?php echo ($_POST['task_type'] ?? 'one_time') === 'one_time' ? 'selected' : ''; ?>>One-Time Task</option>
                            <option value="recurring" <?php echo ($_POST['task_type'] ?? '') === 'recurring' ? 'selected' : ''; ?>>Recurring Task</option>
                        </select>
                    </div>
                    
                    <div class="form-group" id="recurrence_div" style="display: <?php echo ($_POST['task_type'] ?? 'one_time') === 'recurring' ? 'block' : 'none'; ?>;">
                        <label for="recurrence_interval">Recurrence Interval (days)</label>
                        <input type="number" id="recurrence_interval" name="recurrence_interval" min="1" value="<?php echo htmlspecialchars($_POST['recurrence_interval'] ?? ''); ?>">
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="priority">Priority *</label>
                        <select id="priority" name="priority" required>
                            <option value="low" <?php echo ($_POST['priority'] ?? 'medium') === 'low' ? 'selected' : ''; ?>>Low</option>
                            <option value="medium" <?php echo ($_POST['priority'] ?? 'medium') === 'medium' ? 'selected' : ''; ?>>Medium</option>
                            <option value="high" <?php echo ($_POST['priority'] ?? 'medium') === 'high' ? 'selected' : ''; ?>>High</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label for="assigned_to">Assign To User</label>
                        <select id="assigned_to" name="assigned_to">
                            <option value="">Unassigned</option>
                            <?php foreach ($users as $u): ?>
                                <option value="<?php echo $u['id']; ?>" <?php echo (isset($_POST['assigned_to']) && $_POST['assigned_to'] == $u['id']) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($u['full_name'] . ' (' . $u['username'] . ')'); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>
                
                <button type="submit" class="btn btn-primary btn-block">Create Task</button>
            </form>
        </div>
    </div>
</body>
</html>
